/// <summary>
/// Ansteuerung der PortDLL aus C#/dotNET
/// 
/// Diese Klasse enthält Code für die Benutzeroberfläche, sowie
/// den Anwendungsspezifischen Code.
/// 
/// Das Programm ist ein Beispiel dafür, wie man mittels der PortDLL
/// die Leitungen DTR,RTS und TXD schalten, DSR, CTS, DSD 
/// und RI auslesen, sowie einen einfachen Signalgenerator
/// implementieren kann. 
///
/// Es ist sicher nicht als Beispiel für zeitgemäße oder 
/// effiziente Programmierung der Seriellen Schnittstelle bzw. von
/// zeitkritischen Applikationen unter C#/dotNET zu verstehen. 
/// Aber vielleicht hilft es dem einen oder anderen dabei weitere Programme
/// für die PortDLL nach C# zu portieren, die tollen Sachen auf Burkhard
/// Kainkas Seite auszuprobieren oder einfach nur den ein oder anderen
/// Teil des Codes für eigene Programme zu verwenden.
///
/// Der Signalgenerator ist "experimentell". Er schafft auf meinem Rechner
/// bis zu 25KHz. Aufgrund der Einschränkungen (Overhead / absichtlich nicht 
/// abgeschaltetes Multitasking) kann man keine ganz genaue Frequenzvorgabe
/// machen, deshalb wird nicht die Vorgabe, sondern die mittels
/// PerformanceCounter gemessene tatsächliche Frequenz angezeigt.
/// 
/// Wer will kann ja mal einen Lautsprecher z.B. an TXD anschließen und den
/// Signalgenerator auf ca. 20KHz fahren so dass man nur noch die
/// Störungen durch Taskwechsel hört. Mit einiger Übung kann man ganz gut
/// einschätzen was gerade auf dem Rechner los ist (-;
/// 
/// Die von mir implementierte Methode "microsleep()" verursacht
/// eine hohe CPU-Last. Das war aber auf die Schnelle das Beste was ich
/// machen konnte. Hier wäre es schön eine Methode zu haben, die auch 
/// Wartezeit im Microsekunden-Bereich an den Scheduler respektive andere
/// Tasks disponiert. Irgendwo in Kernel32.dll gibt es sowas wohl...
/// Ach ja, Thread.sleep() gibt zwar die CPU frei, aber ist nur bis ca. 30 Hz 
/// zu gebrauchen. Die Funktionen PortDLL.DELAYUS() und PortDLL.TIMEREADUS() 
/// funktionieren auf meinem Rechner bzw. mit meiner port.dll leider nicht.
/// 
/// Das Ganze sollte sich mit Visual C# 2005 Express Edition oder
/// Visual Studio 2005 übersetzen lassen.
/// Die "port.dll" muss sich entweder im Windows Verzeichnis befinden
/// oder da wo die ausführbare Datei dieses Programmes liegt.
///  
/// Sourcen und Programm sind von mir für die freie private Verwendung
/// freigegeben und unterliegen keinen Lizenzbedingungen,
/// außer dass weder eine Haftung dafür übernehme noch 
/// irgendeine Funktionalität garantiere.
/// Die Benutzung erfolgt also auf eigene Gefahr!
///
/// Wer findet dass dieses Programm oder der Source in irgend einer Form
/// nützlich ist, der kann mir ja eine Mail oder eine Postkarte schreiben. 
/// 
/// Also dann, viel Spaß damit!
/// Sascha Bader
/// 
/// Weitere Informationen und/oder aktuellere Versionen sind
/// mit etwas Glück hier zu finden: http://www.sascha-bader.de/html/portdlldemo.html 
/// </summary>

using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Threading;

namespace PortDLLDemo
{
  public partial class PortDLLDemo : Form
  {
    // DLL Imports für die *genaue* Zeitmessung
    [DllImport("Kernel32.dll")]
    private static extern bool QueryPerformanceCounter(
        out long lpPerformanceCount);

    [DllImport("Kernel32.dll")]
    private static extern bool QueryPerformanceFrequency(
        out long lpFrequency);     
    
    // --- Konstanten ---

    // höchster COM-Port der unterstützt und geprüft wird
    const int MAXPORT = 8;     

    // --- Variablen ---

    // gewünschte Frequenz Rechtecksignal Ausgänge
    double outputFreq = 10;    

    // Intervall für eine halbe Periode des Ausgangssignals in Microsekunden
    // (wird berechnet)
    long outputInterval = 10000;

    // Gewünschte Frequenz für aktualisierung der Anzeige 
    // der Eingänge. Es wird eine Festfrequenz
    // verwendet, die nicht zu hoch liegen sollte, da
    // bei höheren Frequenzen die Benutzeroberfläche 
    // das Timing versauen könnte.
    int inputFreq = 10;

    // Zur Messung der tatsächlichen Frequenz
    long lastTime = 0;
    double realInterval = 0;
    long perfCountFreq = 0;

    // Umrechnungsfaktor für den Trackbar 
    // damit auch Frequenzen unter 1 Hz eingestellt werden können
    double trackScale = 1;
    
    // Standard ist COM1
    int com = 0; 
    
    // Status der Ausgänge
    bool DTR = false;
    bool TXD = false;
    bool RTS = false;

    // Toggle der Ausgänge (Signalgenerator)
    bool toggleDTR = false;
    bool toggleTXD = false;
    bool toggleRTS = false;

    // Signalgenerator
    bool sgOn = false;
    
    // Thread für den Signalgenerator
    static Thread sgThread = null;

    /// <summary>
    /// Standard Konstruktor für die GUI-Klasse
    /// </summary>     
    public PortDLLDemo()
    {
      InitializeComponent();
    }

    /// <summary>
    /// Wird beim Öffnen des Forms zur Initialisierung aufgerufen.
    /// Öffnet die Schnittstelle und initialisiert Benutzungsoberfläche.
    /// Zuerst wird COM1 probiert und wenn das nicht klappt dann COM2
    /// </summary>
    /// <param name="sender">Quelle des Events</param>
    /// <param name="e">Argumente des Events</param>
    /// <returns>logic address</returns>
    /// 
    private void Form1_Load(object sender, EventArgs e)
    {
      for (int n = 0; n < MAXPORT - 1; n++)
      {
        comboBoxPort.Items.Add("COM "+(n+1));
      }
      for (int n=0;n<MAXPORT-1;n++)
      {
        if (open(n+1))
        {
          comboBoxPort.SelectedIndex=n;
          break;
        }
      }
      if (com == 0)
      {
        labelStatus.Text = "Can't open any port!";        
      }

      initGUI();

      // feststellen mit welcher Frequenz der Performance Counter läuft
      QueryPerformanceFrequency(out perfCountFreq);

      // setzen der Festfrequenz für Aktualisierungd er Eingänge
      timerInput.Interval = 1000 / inputFreq;
      timerInput.Start();
            
      // setzen der Ausgangsfrequenz
      setOutputFreq(outputFreq);                        
    }

    /// <summary>
    /// Wird beim Schließen des Forms aufgerufen z.B. um Resourcen 
    /// freizugeben.
    /// Die COM Schnittstelle wird wieder geschlossen.
    /// </summary>
    /// <param name="sender">Quelle des Events</param>
    /// <param name="e">Argumente des Events</param>
    /// 
    private void Form1_FormClosed(object sender, FormClosedEventArgs e)
    {
      stopSignalGeneration();
      close();
    }

    /// <summary>
    /// Wird aufgerufen wenn die DTR-Checkbox betätigt wird.
    /// </summary>
    /// <param name="sender">Quelle des Events</param>
    /// <param name="e">Argumente des Events</param>
    /// 
    private void checkBoxDTR_CheckedChanged(object sender, EventArgs e)
    {
      DTR = checkBoxDTR.Checked;
      setOutputs();
    }

    /// <summary>
    /// Wird aufgerufen wenn die RTS-Checkbox betätigt wird
    /// </summary>
    /// <param name="sender">Quelle des Events</param>
    /// <param name="e">Argumente des Events</param>
    /// <returns>logic address</returns>
    /// 
    private void checkBoxRTS_CheckedChanged(object sender, EventArgs e)
    {
      RTS = checkBoxRTS.Checked;
      setOutputs();
    }

    /// <summary>
    /// Wird aufgerufen wenn die TXD-Checkbox betätigt wird    
    /// </summary>
    /// <param name="sender">Quelle des Events</param>
    /// <param name="e">Argumente des Events</param>
    /// <returns>logic address</returns>
    ///
    private void checkBoxTXD_CheckedChanged(object sender, EventArgs e)
    {
      TXD = checkBoxTXD.Checked;
      setOutputs();
    }

    
    /// <summary>
    /// Aktualisere die GUI bzw. Anzeige der Eingänge.
    /// Wird vom TimerInput aufgerufen.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void timerInput_Tick(object sender, EventArgs e)
    {
      updateGUI();
    }


    /// <summary>
    /// Wird aufgerufen wenn über den TrackBar eine neue
    /// Frequenz eingestellt wird.
    /// Es wird eine nichtlineare Skala verwendet um
    /// einfacher hohe Frequenzen einstellen zu können
    /// </summary>
    /// <param name="sender">Quelle des Events</param>
    /// <param name="e">Argumente des Events</param>
    /// <returns>logic address</returns>
    ///     
    /// <returns>logic address</returns>  
    private void trackBarFreq_Scroll(object sender, EventArgs e)
    {
      setOutputFreq(Math.Pow(trackBarFreq.Value,1.25));
    }

    /// <summary>
    /// Berechne und setzte das Mikrosekunden Intervall für das Ausgangssignal.
    /// </summary>
    /// <param name="newFreq"></param>
    private void setOutputFreq(double newFreq)
    {
      outputFreq=newFreq;
      outputInterval = (long)(500000 / outputFreq);
    }

    /// <summary>
    /// Wird aufgerufen wenn die Checkbox checkBoxTXDblink geändert wird
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void checkBoxTXDblink_CheckedChanged(object sender, EventArgs e)
    {
      toggleTXD = checkBoxTXDblink.Checked;
    }

    /// <summary>
    /// Wird aufgerufen wenn die Checkbox checkBoxRTSblink geändert wird
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void checkBoxRTSblink_CheckedChanged(object sender, EventArgs e)
    {
      toggleRTS = checkBoxRTSblink.Checked;
    }

    /// <summary>
    /// Wird aufgerufen wenn die Checkbox checkBoxDTRblink geändert wird
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void checkBoxDTRblink_CheckedChanged(object sender, EventArgs e)
    {
      toggleDTR = checkBoxDTRblink.Checked;
    }

    /// <summary>
    /// Gleicht initiale Werte mit GUI ab
    /// </summary>
    private void initGUI()
    {
      checkBoxDTRblink.Checked = toggleDTR;
      checkBoxTXDblink.Checked = toggleTXD;
      checkBoxRTSblink.Checked = toggleRTS;
      checkBoxGenerateSignal.Checked = sgOn;
      
      trackBarFreq.Value = (int) (trackScale * outputFreq);
      updateGUI();
    }


    /// <summary>
    /// Aktualisiert die GUI mit den Eingangssignalen
    /// </summary>
    private void updateGUI()
    {
      // update GUI
      checkBoxDCD.Checked = PortDLL.DCD();
      checkBoxDSR.Checked = PortDLL.DSR();
      checkBoxRI.Checked = PortDLL.RI();
      checkBoxCTS.Checked = PortDLL.CTS();

      checkBoxTXD.Enabled = !sgOn || !toggleTXD ;
      if (checkBoxTXD.Enabled)
      {
        checkBoxTXD.Checked = TXD;
      }
      checkBoxDTR.Enabled = !sgOn || !toggleDTR;
      if (checkBoxDTR.Enabled)
      {
        checkBoxDTR.Checked = DTR;
      }
      checkBoxRTS.Enabled = !sgOn || !toggleRTS;
      if (checkBoxRTS.Enabled)
      {
        checkBoxRTS.Checked = RTS;
      }
      
      labelFreqReal.Text = String.Format("{0:F1} Hz", (1000.0 / realInterval ) );
    }


    /// <summary>
    /// Öffne den angegebenen COM-Port
    /// Ist bereits ein COM-Port geöffnet,
    /// so wird dieser vorher geschlossen
    /// </summary>
    /// <param name="port"></param>
    /// <returns>true wenn port geöffnet</returns>
    public bool open(int port)
    {
      bool ret = false;
      try
      {
        if (com != 0)
        {
          PortDLL.CLOSECOM();
        }
        if (PortDLL.OPENCOM("COM" + port + ":1200,n,8,1"))
        {
          ret = true;
          com = port;
          labelStatus.Text = "OK";
        }
        else
        {
          labelStatus.Text = "Problem with port " + port + "!";
        }
      }
      catch (Exception x)
      {
        labelStatus.Text = "Problem with port " + port + "!";
      }

      // sicher stellen dass keine Ports den definierten Pegel haben, 
      // denn Windows spielt (zumindest bei mir) beim Öffnen der Ports
      // gerne mal an den Leitungen herum
      if (ret)
      {
        setOutputs();
      }
      return ret;
    }
    
    /// <summary>
    /// Schliesst alle offenen COM-Ports
    /// </summary>
    public void close()
    {
      PortDLL.CLOSECOM();
    }

     /// <summary>
     /// Setzt die Ausgabeleitungen auf die vorgegebenen Pegel
     /// </summary>
    public void setOutputs()
    {
      PortDLL.DTR(DTR ? 1 : 0);
      PortDLL.RTS(RTS ? 1 : 0);
      PortDLL.TXD(TXD ? 1 : 0);
    }

    /// <summary>
    /// Wird aufgerufen wenn ein COM-Port ausgewählt wird
    /// Versucht den Port zu öffnen.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void comboBoxPort_SelectedIndexChanged(object sender, EventArgs e)
    {      
      open(comboBoxPort.SelectedIndex+1);      
    }

    /// <summary>
    /// Wird aufgerufen wenn jemand auf den "info"-Link klickt
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void linkLabel1_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
    {
      AboutBox1 about = new AboutBox1();
      about.ShowDialog();
    }

    /// <summary>
    /// Wird aufgerufen wenn der SIgnalgenerator ein oder aus geschaltet wird
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void checkBoxGenerateSignal_CheckedChanged(object sender, EventArgs e)
    {
      sgOn = checkBoxGenerateSignal.Checked;
      if (sgOn == true)
      {
        startSignalGeneration();
      }
      else
      {
        stopSignalGeneration();
      }
    }


    /// <summary>
    /// Beginne mit der Signalerzeugung
    /// </summary>
    private void startSignalGeneration()
    {
      stopSignalGeneration();
      sgThread = new Thread(new ThreadStart(generateSignal));
		  sgThread.Start();			
    }

    /// <summary>
    /// Beende die Signalerzeugung
    /// </summary>
    private void stopSignalGeneration()
    {
      // Abbrechen der Signalerzeugung (auf die harte Tour (-; )
      if (sgThread != null)
      {
        sgThread.Abort();  
        sgThread = null;
      }

      // Buttons wieder freischalten
      updateGUI();
    }


    /// <summary>
    /// Erzeuge periodisch ein Ausgangssignal
    /// Läuft in einem eigenen Thread wenn der 
    /// Signalgenerator aktiv ist
    /// </summary>
    private void generateSignal()
    {

      while (true)
      {
        //PortDLL.TIMEINIT();

        //PortDLL.DELAY(outputInterval/2);
        //Thread.Sleep(outputInterval/2);
        microsleep(outputInterval);

        toggleSignals();

        // tatsächliches Intervall berechnen
        long currentTime = 0;
        QueryPerformanceCounter(out currentTime);
        realInterval = ((double)1000 * (currentTime - lastTime) / (double)perfCountFreq);
        lastTime = currentTime;
        

        //PortDLL.DELAY(outputInterval/2);
        //Thread.Sleep(outputInterval/2);
        microsleep(outputInterval);

        toggleSignals();
      }
    }

    
    /// <summary>
    /// Schaltet die Ausgänge periodisch um
    /// </summary>
    private void toggleSignals()
    {
      // Pegel umschalten

      if (toggleDTR)
      {
        DTR = !DTR;
      }
      if (toggleRTS)
      {
        RTS = !RTS;
      }
      if (toggleTXD)
      {
        TXD = !TXD;
      }

      // und ausgeben
      setOutputs();
    }


     /// <summary>
     /// Wartet die angegebene Zahl von Mikrosekunden.
     /// Achtung diese Routine erzeugt 100% CPU-Last, 
     /// ebenso wie es PortDLL.DELAY() tun würde, welche aber,
     /// ebenso wie Thread.sleep() nicht die nötige Auflösung bringt.
     /// PortDLL.DELAYUS() funktionier ja leider auch nicht...
     /// So muss es eben bis auf weiteres diese Funktion sein.
     /// Wenn jemand ein "microsleep" für dotNET kennt, das
     /// keine 100% CPU-Last erzeugt, dann wäre ich für einen
     /// entsprechenden Hinweis dankbar.
     /// </summary>
     /// <param name="microsecs"></param>
    private void microsleep(long microsecs)
    {
      long currentTime = 0;
      QueryPerformanceCounter(out currentTime);

      long waitForTime = currentTime + (long)(microsecs * perfCountFreq / 1000 / 1000);

      do
      {
        QueryPerformanceCounter(out currentTime);
      }
      while (currentTime < waitForTime);
    }

  }
}
